import type { NextAuthOptions } from "next-auth";
import CredentialsProvider from "next-auth/providers/credentials";
import bcrypt from "bcrypt";
import { prisma } from "@/lib/prisma";

export const authOptions: NextAuthOptions = {
  session: { strategy: "jwt" },

  pages: {
    signIn: "/login",
  },

  providers: [
    CredentialsProvider({
      name: "credentials",
      credentials: {
        email: { label: "Email", type: "email" },
        password: { label: "Mot de passe", type: "password" },
      },

      async authorize(credentials) {
        const email = credentials?.email?.toLowerCase().trim();
        const password = credentials?.password ?? "";

        if (!email || !password) return null;

        const user = await prisma.user.findUnique({
          where: { email },
        });

        if (!user) return null;

        const ok = await bcrypt.compare(password, user.password);
        if (!ok) return null;

        // 👇 CE QUI EST RETOURNÉ ICI VA DANS jwt()
        return {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role,
          phone: user.phone,
          avatar: user.avatar,
        };
      },
    }),
  ],

  callbacks: {
    /* ===== JWT ===== */
    async jwt({ token, user }) {
      if (user) {
        token.uid = (user as any).id;
        token.role = (user as any).role;
        token.phone = (user as any).phone ?? null;
        token.avatar = (user as any).avatar ?? null;
      }
      return token;
    },

    /* ===== SESSION ===== */
    async session({ session, token }) {
      if (session.user) {
        (session as any).user.id = token.uid;
        (session as any).user.role = token.role;
        (session as any).user.phone = token.phone;
        (session as any).user.avatar = token.avatar;
      }
      return session;
    },
  },
};
